package com.colectivosvip.schindler.service;

import java.util.ArrayList;
import java.util.List;

import android.app.IntentService;
import android.content.Context;
import android.content.Intent;
import android.location.Location;
import android.location.LocationManager;
import android.os.Bundle;
import android.os.SystemClock;
import android.util.Log;
import es.javocsoft.android.lib.toolbox.ToolBox;
import com.colectivosvip.schindler.ApplicationBase;
import com.colectivosvip.schindler.Constants;
import com.colectivosvip.schindler.receiver.LocationChangedReceiver;
import com.colectivosvip.schindler.receiver.proximity.LocationProximityChangedReceiver;
import com.colectivosvip.schindler.sqllite.SQLite;
import com.colectivosvip.schindler.sqllite.tables.DBLocation;

/**
 * When localization changes and is better than previous one this service
 * is called in order to do something with the location.
 * <br><br>
 * The service (an IntentService) does some work. When it is finished, it 
 * releases the wake lock by calling completeWakefulIntent(intent). The 
 * intent it passes as a parameter is the same intent that the 
 * LocationChangedReceiver originally passed in.
 * <br><br>
 * NOTES:<br><br>
 * Remember that if you need to show something in the UI you must deliver
 * a broadcast to some receiver and do the UI operations there.
 * 
 * @author JavocSoft 2015
 * @version 2.0<br>
 * $Rev: 875 $<br>
 * $LastChangedDate: 2016-12-15 18:50:05 +0100 (jue, 15 dic 2016) $<br>
 * $LastChangedBy: jgonzalez $
 *
 */
public class LocationChangeWakefulService extends IntentService {

	public static final String CALLING_SOURCE = "CALLING_SOURCE";
	public static final String CALLING_SOURCE_PROXIMITY_SERVICE = "LocationProximityChangedReceiver";
	public static final String CALLING_SOURCE_LOCATION_SERVICE = "LocationChangedReceiver";
	
	public static final String LOCATION_KEY = "location";
	
	public LocationChangeWakefulService() {
		super("Location Change Action Service");
	}
	
	public LocationChangeWakefulService(String name) {
		super(name);		
	}

	@Override
	protected void onHandleIntent(Intent intent) {
		
		if(ApplicationBase.debugMode)
			Log.d(Constants.TAG, "LocationChangeWakefulService: START");
		
		// At this point, this service calling Receiver is still holding a wake lock
        // for us.  We can do whatever we need to here and then tell it that
        // it can release the wakelock.
        //
        // Note that when using this approach you should be aware that if your
        // service gets killed and restarted while in the middle of such work
        // (so the Intent gets re-delivered to perform the work again), it will
        // at that point no longer be holding a wake lock since we are depending
        // on the caller Wakeful Receiver to that for us.  If this is a concern, 
		// you can acquire a separate wake lock here.
        updateWithNewLocation(getApplicationContext(), intent);
        
        if(ApplicationBase.debugMode)
        	Log.d(Constants.TAG, "LocationChangeWakefulService: Finished @ " + SystemClock.elapsedRealtime());
        
        if(intent.getExtras().getString(CALLING_SOURCE)!=null &&
    			intent.getExtras().getString(CALLING_SOURCE).equals(CALLING_SOURCE_PROXIMITY_SERVICE)){
        	LocationProximityChangedReceiver.completeWakefulIntent(intent);
        }else{
        	LocationChangedReceiver.completeWakefulIntent(intent);
        }	
	}

	
	//AUXILIAR
	
	
	/**
     * This method is used to do something with the location.
     *
     * @param location
     */
    private void updateWithNewLocation(final Context context, Intent intent) {
    	Bundle extras = intent.getExtras();
    	Location location = extras.getParcelable(LOCATION_KEY);
        if(location!=null)  {
        	ApplicationBase.location = new ApplicationBase.LocationInfo(extras);
        	ApplicationBase.locationLast = new ApplicationBase.LocationInfo(extras);
        	if(location.getProvider()!=null)
        		ApplicationBase.isGPSProviderEnabled = location.getProvider().equals(LocationManager.GPS_PROVIDER);
        	
            //You can check the location in http://itouchmap.com/latlong.html
        	
        	if(ApplicationBase.debugMode){
        		//Location information print
        		try{
	        		Log.d(Constants.TAG, "Location: Lat: " + location.getLatitude() + " Lon: " + location.getLongitude() + ". GPS Enabled: " + ApplicationBase.isGPSProviderEnabled);
	        		Log.d(Constants.TAG, "Provider: " +  location.getProvider() );
	        		Log.d(Constants.TAG, "GPS Enabled?: " +  location.getProvider().equals(LocationManager.GPS_PROVIDER) );
	        		
	        		Log.d(Constants.TAG, "Country: " + ApplicationBase.location.getCountry());
	        		Log.d(Constants.TAG, "Country Code: " + ApplicationBase.location.getCountryCode());
	        		Log.d(Constants.TAG, "City: " + ApplicationBase.location.getCity());
	        		Log.d(Constants.TAG, "Address: " + ApplicationBase.location.getAddress());
	        		Log.d(Constants.TAG, "Postal Code: " + ApplicationBase.location.getPostalCode());
        		}catch(Exception e){
        			Log.d(Constants.TAG, "Error showing location information (" + e.getMessage() + ").", e);
        		}
        	}
            
        	//We capture any error so service can finish and log the error.
        	try{
	        	if(ApplicationBase.enableSQLite) {
	        		if(extras.getString(CALLING_SOURCE)!=null &&
	            			extras.getString(CALLING_SOURCE).equals(CALLING_SOURCE_PROXIMITY_SERVICE)){
	        			//To be able to remind the locations
	        			doLocationChangeJobWithDBEnabled(context, intent, location);
	        		}else{
	        			doLocationChangeJobWithoutDBEnabled(context, intent, location);
	        		}       		
	        	}else{
	        		doLocationChangeJobWithoutDBEnabled(context, intent, location);
	        	}       	
        	}catch(Exception e){
        		Log.d(Constants.TAG, "Error processing location update (" + e.getMessage() + ").", e);
        	}
        }
    }
    
    /**
     * Receives the location change and do job.
     * 
     * @param context
     * @param intent
     * @param location
     */
    private void doLocationChangeJobWithoutDBEnabled(final Context context, Intent intent, Location location) {
    	//In this case we do not have a DB to store previous locations so we can not
    	//exclude locations given user option.
    	doJob(context, intent, true, null, location.getLatitude(), location.getLongitude());
    }
    
    /**
     * Do checks on location changes and decide if do something or
     * not.<br><br>
     * We check if the current location is already in the radius of 
     * a previous one in the DB and see if the user decided to not 
     * be reminded when entering into that location radius.
     * <br><br> 
     * Also time between notifications is check.
     *  
     * @param context
     * @param intent
     * @param location
     */      
    private void doLocationChangeJobWithDBEnabled(final Context context, Intent intent, Location location) {
    	
    	//synchronized (this) {
	    	long dbLocationId = -1;
	    	
	    	//We must be notified if we change the location going outside of the minDistance.        		
			DBLocation lastLoc = SQLite.getInstance(context).getDBHelper().getLastLocation();
	    	if(lastLoc!=null){
	    		//Means that the new location is the radius of a already 
	    		//stored in DB location.
	    		LocationCheck locCheck = locationCheck(context, location);
	    	    	
	    		//We avoid multiple notifications if the providers inform at the same time.
	    		/*Calendar now = Calendar.getInstance();
	        	now.setTime(new Date());
	        	Calendar lastLocationTs = Calendar.getInstance();
	        	lastLocationTs.setTimeInMillis(lastLoc.ts);
	        	lastLocationTs.add(Calendar.MINUTE, ApplicationBase.LOCATION_ALERT_TIME_THRESHOLD);*/
	       		
	        	//if(now.after(lastLocationTs)){
	        		if(!locCheck.exists) {
	        			//We moved more than the minimal distance threshold	        			
	        			dbLocationId = SQLite.getInstance(context).getDBHelper().insertLocation(context, location, false);						
	        	   	}else{
	        	   		dbLocationId = lastLoc.id;	        	   		
	        	   		SQLite.getInstance(context).getDBHelper().updateLocation(context, lastLoc, location);	        	   		
	        	   	}
	        		doJob(context, intent, locCheck.isRemindAllowed, dbLocationId, location.getLatitude(), location.getLongitude());
	        	//}    		
	    	}else{
	    		dbLocationId = SQLite.getInstance(context).getDBHelper().insertLocation(context, location, false);
	    		doJob(context, intent, true, dbLocationId, location.getLatitude(), location.getLongitude());
	    	}
		//}    	
    }
    
    /**
     * Do the task.
     * 
     * @param context
     * @param intent
     */
    private void doJob(final Context context, Intent intent, boolean remindAllowed, Long dbLocationId, double lat, double lng) {
    	Bundle extras = intent.getExtras();
    	
    	if(ApplicationBase.debugMode){
			Log.i(Constants.TAG, "Location change. You can be notified!");	        					        				
		}
    	
    	//Inform to the Web    	
    	if(extras.getString(CALLING_SOURCE)==null || 
    			(extras.getString(CALLING_SOURCE)!=null && 
    			extras.getString(CALLING_SOURCE).equals(CALLING_SOURCE_LOCATION_SERVICE))){
    		
    		boolean isLastAndroidKnownLocation = false;
    		if(extras.getInt(Constants.LOCATION_IS_LAST_KNOWN_KEY)==1){
    			//In this case we inform to the web this last known location but 
    			//the service is not stopped and continues until current location is
    			//found.
    			isLastAndroidKnownLocation = true;
    		}    		
    		ApplicationBase.appWebJSInterface.androidAppInformLocationToWeb(false, isLastAndroidKnownLocation);
    	}
    	
    	if(remindAllowed) {    		
    		//Do some tasks related to a location change.
    		doLocationRelatedTask(context, extras, dbLocationId, lat, lng);
    		
    		/*if(ApplicationBase.debugMode){    			
	    		Integer notificationId = null;
	    		List<android.support.v4.app.NotificationCompat.Action> actions = null;
	    		//Prepare actions of the notification
			    notificationId = 88; //We always use the same so last one is updated.
			    actions = prepareNotificationActions(context, notificationId, dbLocationId);
	    		
		    	//Create the notification
		    	ApplicationBase.generateSystemNotification(context, 
		    			ToolBox.NOTIFICATION_STYLE.NORMAL_STYLE, ToolBox.NOTIFICATION_PRIORITY.DEFAULT,
		    			true, null,
		    			null,
		    			"Location changed", "Your location changed!", "Receiving new location...",
		    			null, 
		    			null, 
		    			null, null,
		    			null, 
		    			null, null,
		    			null, null, true, extras, notificationId, actions);
    		}*/
	    	
    	}else{
    		if(ApplicationBase.debugMode)
    			Log.d(Constants.TAG, "Location notifications banned ('Not Remind' option set)");
    	}
    }
    
    
    private void doLocationRelatedTask(final Context context, Bundle extras, Long dbLocationId, double lat, double lng) {
    	if(extras.getString(CALLING_SOURCE)!=null &&
    			extras.getString(CALLING_SOURCE).equals(CALLING_SOURCE_PROXIMITY_SERVICE)){
    		
    		if(ApplicationBase.debugMode)
    			Log.d(Constants.TAG, "LocationChangeWakefulService: doLocationRelatedTask.");
    		
    		//ProximityExternalApiTask.queryForProximityOffers(context, lat, lng, dbLocationId, extras);
    		Log.i(Constants.TAG,"Proximity change received. There is nothing to do specified.");
    	}
    }
    
    /**
     * Prepare the notification actions.
     * 
     * @param context
     * @param notificationId
     * @return
     */
    private List<android.support.v4.app.NotificationCompat.Action> prepareNotificationActions (
    		final Context context,
    		Integer notificationId,
    		Long dbLocationId) {
    	
    	List<android.support.v4.app.NotificationCompat.Action> actions = null;
    	if(ApplicationBase.enableSQLite && dbLocationId!=null) {
	    	//
	    	//Actions:
	    	//
	    	//Do not remind				
			Bundle extras = new Bundle();
			extras.putInt(LocationChangedReceiver.LOCATION_NOTIFICATION_KEY_ACTION, LocationChangedReceiver.LOCATION_NOTIFICATION_ACTION_NOT_REMIND);
			extras.putLong(LocationChangedReceiver.LOCATION_NOTIFICATION_KEY_LOCATIONID, dbLocationId);
			android.support.v4.app.NotificationCompat.Action actionDoNotRemind = 
				ToolBox.notification_createActionButton(context, 
						ToolBox.NOTIFICATION_ACTION_TARGET_TYPE.RECEIVER,
						LocationChangedReceiver.ACTION_LOCATION_NOTIFICATION, 
						LocationChangedReceiver.class, 
						LocationChangedReceiver.LOCATION_NOTIFICATION_KEY_NOTID, notificationId, 
						android.R.drawable.ic_lock_silent_mode, "Not Remind", extras);
			//Configure		
			extras = new Bundle();
			extras.putInt(LocationChangedReceiver.LOCATION_NOTIFICATION_KEY_ACTION, LocationChangedReceiver.LOCATION_NOTIFICATION_ACTION_CONFIGURE);
			extras.putLong(LocationChangedReceiver.LOCATION_NOTIFICATION_KEY_LOCATIONID, dbLocationId);
			android.support.v4.app.NotificationCompat.Action actionConfigurar = 
				ToolBox.notification_createActionButton(context,
						ToolBox.NOTIFICATION_ACTION_TARGET_TYPE.RECEIVER,
						LocationChangedReceiver.ACTION_LOCATION_NOTIFICATION, 
						LocationChangedReceiver.class, 
						LocationChangedReceiver.LOCATION_NOTIFICATION_KEY_NOTID, notificationId, 
						android.R.drawable.ic_menu_preferences, "Configure", extras);
			//Close
			/*extras = new Bundle();
			extras.putInt(LocationChangedReceiver.LOCATION_NOTIFICATION_KEY_ACTION, LocationChangedReceiver.LOCATION_NOTIFICATION_ACTION_CLOSE);
			extras.putLong(LocationChangedReceiver.LOCATION_NOTIFICATION_KEY_LOCATIONID, dbLocationId);
			android.support.v4.app.NotificationCompat.Action actionClose = 
				ToolBox.notification_createActionButton(context,
						ToolBox.NOTIFICATION_ACTION_TARGET_TYPE.RECEIVER,
						LocationChangedReceiver.ACTION_LOCATION_NOTIFICATION, 
						LocationChangedReceiver.class, 
						LocationChangedReceiver.LOCATION_NOTIFICATION_KEY_NOTID, notificationId, 
						android.R.drawable.ic_notification_clear_all, "Close", extras);
			*/
			//
			//Add the actions to a list that is passed to the notification
	    	actions = new ArrayList<android.support.v4.app.NotificationCompat.Action>();
	    	actions.add(actionDoNotRemind);
	    	//actions.add(actionClose);
	    	actions.add(actionConfigurar);    	
    	}
    	
    	return actions;
    }
    
    /**
     * Looks into registered locations to check if the user decided to not be
     * reminded about the location.
     * <br><br>
     * The method gets the distance between points with haversine formula and if
     * the distance is less than the threshold distance, the location is already
     * in the DB.
     * 
     * @param context
     * @param location	The location to check.
     * @param locFound	If the location is already in the database, the method sets this 
     * 					parameter to TRUE.
     * @return
     */
    private LocationCheck locationCheck(final Context context, Location location) {
    	LocationCheck res = new LocationCheck();
    	
    	//Get locations stored in the DB
    	List<DBLocation> locations = SQLite.getInstance(context).getDBHelper().getLocations();
    	double distanceBetweenLocations = 0;
    	for(DBLocation loc:locations) {
    		distanceBetweenLocations = ToolBox.location_distance(loc.lat, loc.lng, 
    				location.getLatitude(), location.getLongitude());
    		if(distanceBetweenLocations<ApplicationBase.LOCATION_UPDATE_MIN_DISTANCE){
    			res.exists = true;
    			if(loc.notRemind==1) {
    				res.isRemindAllowed = false;    				
    			}
    			break;
    		}
    	}
    	
    	return res;
    }
    
    class LocationCheck {
    	
    	public boolean isRemindAllowed = true;
    	public boolean exists;
    	
    	public LocationCheck() {}    	
    	
    }
    
}
