package com.colectivosvip.schindler.javascript;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Collection;

import android.content.Context;
import android.os.Build;
import android.util.Log;
import android.webkit.JavascriptInterface;
import android.webkit.ValueCallback;

import com.google.gson.Gson;

import es.javocsoft.android.lib.toolbox.ToolBox;
import es.javocsoft.android.lib.toolbox.gcm.NotificationModule;
import es.javocsoft.android.lib.toolbox.json.GsonProcessor;
import es.javocsoft.android.lib.toucan.client.ToucanClient;
import com.colectivosvip.schindler.ApplicationBase;
import com.colectivosvip.schindler.ApplicationBase.LocationInfo;
import com.colectivosvip.schindler.Constants;
import com.colectivosvip.schindler.PrincipalActivity;
import com.colectivosvip.schindler.R;
import com.colectivosvip.schindler.external.ProximityExternalApiTask;


/**
 * This class is an Interface to be able to launch Android native
 * methods from web through JS. The channel is bidirectional and
 * allows to call web JS methods within Android application.<br><br>
 *
 * To enable it in the WebView:<br><br>
 * 
 * //This enables the JS in the webviews's content.<br>
 * myWebView.getSettings().setJavaScriptEnabled(true);<br>
 * //This enables to expose to webviews's web some native android app methods.<br>
 * myWebView.addJavascriptInterface(new WebAppJSInterface(this), "Android");<br><br>
 * 
 * Once enabled, in web side there should be a JS object called "Android".<br><br>
 * 
 * We can use it to check if we are running the web inside Android native app
 * by using this JS code:<br>
 * <code>
 * 	if("Android" in window){....} 
 * </code>
 *
 * <br><br>
 * See <a href="http://developer.android.com/guide/webapps/webview.html">WebView</a>.
 *
 * @author JavocSoft, 2014
 * @version 1.0<br>
 * $Rev: 884 $<br>
 * $LastChangedDate: 2017-01-25 15:00:19 +0100 (mié, 25 ene 2017) $<br>
 * $LastChangedBy: jgonzalez $
 *
 */    
public class WebAppJSInterface {
    
	private Context mContext;
    private PrincipalActivity pActivity;
    

    /** Instantiate the interface and set the context */
    public WebAppJSInterface(PrincipalActivity pActivity) {
        this.mContext = pActivity.getApplicationContext();
        this.pActivity = pActivity;
    }

    
    //PUBLIC JS methods available in a web running
    //inside a WebView.
    
    @JavascriptInterface
    public void androidAppScanCode() {
    	if(ApplicationBase.debugMode)
    		Log.d(Constants.TAG + "::WebJSInterface", "Scan code command send from web.");
    	
    	if(!ApplicationBase.enableQR) {
    		if(ApplicationBase.debugMode)
    			Log.d(Constants.TAG, "Scan code module not enabled.");
    		//final String jsonResult = urlEncode("{\"resultCode\":\"module_not_enabled\",\"codeType\":\"none\"}");
    		
    		//We have to load in the webview thread
    		ToolBox.webview_runJavascript(pActivity.getWebView(), "javascript:" + Constants.VV_JS_SCANCODERESULT_METHOD + "(null)", true, null);    		
    	}else{
    		if(ApplicationBase.enableQRTooltipDialog && !(Boolean)ToolBox.prefs_readPreference(mContext, Constants.PREF_NAME, Constants.PREF_KEY__SHOWSCANTIP, Boolean.class)) {
    			pActivity.createScanCodeTipDialog();
    		}else{
        		//ZXing Barcode Scanner
        		pActivity.initiateCodeScan();
        	}        		
    	}
    }
    
    /**
     * This method tries to get a list of domains that can be accessed via webview. Any other
     * URL that is not from a valid domain should be opened in a separate navigator window.
     */
    @JavascriptInterface
    public void androidAppGetAvailableDomains() {
    	if(ApplicationBase.debugMode)
    		Log.d(Constants.TAG + "::WebJSInterface", "Get available domains from web.");
    	
    	ToolBox.webview_runJavascript(pActivity.getWebView(), "javascript:" + Constants.VV_JS_GETAVAILABLEDOMAINS_METHOD + "()", true, new ValueCallback<String>() {
			
			@Override
			public void onReceiveValue(String value) {
				if(value!=null && !value.equalsIgnoreCase("null")){
					if(ApplicationBase.debugMode)
			    		Log.d(Constants.TAG + "::WebJSInterface", "Get allowed domains <<" + value + ">> from web.");
					
					try {
						value = URLDecoder.decode(value, "UTF-8");
						//FIX. We remove extra double quotes added by the JS channel.
						if(value.startsWith("\"") && value.endsWith("\""))
							value = value.substring(1, value.length()-1);
						
						//Process JSON to get a valid list of URLs
						GsonProcessor gsonP = GsonProcessor.getInstance();
						Gson gson = gsonP.getGson(GsonProcessor.GSON_PROCESSOR_TYPE.GSONP);						
						DomainsInfo dInfo = gson.fromJson(value, DomainsInfo.class);
						ApplicationBase.addURL(dInfo.entry_points);
						Log.d(Constants.TAG + "::WebJSInterface", "Allowed domains get: " + (dInfo.entry_points!=null?dInfo.entry_points.size():"none") + ".");
						
					} catch (UnsupportedEncodingException e) {
						Log.e(Constants.TAG + "::WebJSInterface", "Get allowed domains error [" + e.getMessage() + "] from web.", e);
					} catch (Exception e) {
						Log.e(Constants.TAG + "::WebJSInterface", "Get allowed domains error [" + e.getMessage() + "] from web.", e);
					}					
				}
			}
		});
    }

    public class DomainsInfo {
    	private Collection<String> entry_points;

        public Collection<String> getEntry_points ()
        {
            return entry_points;
        }

        public void setEntry_points (Collection<String> entry_points)
        {
            this.entry_points = entry_points;
        }
    }

    @JavascriptInterface
    public void androidAppRecoverUserGPushId() {
    	if(ApplicationBase.debugMode)
    		Log.d(Constants.TAG + "::WebJSInterface", "Get GCM Registration-ID command send from web.");
    	
        //Get Google Push Service device Id.
        // https://developer.android.com/google/gcm/index.html
        String gPushId = "none";
        String result = null;
        if(!ApplicationBase.enablePush) {
        	if(ApplicationBase.debugMode)
        		Log.d(Constants.TAG, "GCM module not enabled.");
        	gPushId = "none";
        	result = "{\"resultCode\":\"module_not_enabled\",\"gPushId\":\"" + gPushId + "\"}";            	
        	
        }else{
        	gPushId = NotificationModule.getRegistrationId(NotificationModule.APPLICATION_CONTEXT);
        	if(gPushId!=null && gPushId.length()==0) {
        		gPushId = "none";            		
        	}
        	result = "{\"resultCode\":\"module_enabled\",\"gPushId\":\"" + gPushId + "\"}";
        }
        
        final String jsonResult = urlEncode(result);
        //We have to load in the webview thread
        ToolBox.webview_runJavascript(pActivity.getWebView(), "javascript:" + Constants.VV_JS_GPUSHID_INFORM_METHOD + "('" + jsonResult + "')", true, null);
        
        //Send the GCM registration id to the web.
        //(Make the JS call asynchronously)
        //See:
        //	https://developer.android.com/reference/android/webkit/WebView.html#evaluateJavascript(java.lang.String, android.webkit.ValueCallback<java.lang.String>)
        /*pActivity.getWebView().evaluateJavascript("javascript:" + Constants.VV_JS_GPUSHID_INFORM_METHOD + "('" + jsonResult + "')", new ValueCallback<String>() {
			
			@Override
			public void onReceiveValue(String value) {
				//Do something with the result if there is one.
			}
		});*/
    }

    @JavascriptInterface
    public void androidAppInformLocationToWeb(boolean setToNone, final boolean isAndroidLastKnownLocation) {
    	if(ApplicationBase.appIsDestroyed)
    		return;
    	
    	//We only do this if we are in a page with location enabled
    	if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
    		//To enable for all versions we should add a method in the 
    		//page that tell to the page for the location.
	    	if(!ApplicationBase.isPageWithLocationBehavior)
	    		return;
    	}else{
    		//TODO In older versions we should avoid in other way
    	}
    	
    	if(ApplicationBase.debugMode)
    		Log.d(Constants.TAG + "::WebJSInterface", "Sending current location to the web.");
    	
    	//setAndroidAppLocation(address, lat, lng, country)
    	String result = "null,null,null,null";
    	if(ApplicationBase.enableLocation 
    			&& ApplicationBase.locationServiceEnabled && ApplicationBase.location != null
    			&& !setToNone) {
    		
    		String address = null;
    		if(ApplicationBase.location.getAddress()==null || 
    			(ApplicationBase.location.getAddress()!=null && ApplicationBase.location.getAddress().length()==0)){
    			//In case Android could not determine the address, we return an string.
    			address = mContext.getResources().getString(R.string.location_address_not_available);
    		}else{
    			address = ApplicationBase.location.getAddress() +
    					(ApplicationBase.location.getCity()!=null?" " + ApplicationBase.location.getCity():"") +
    					(ApplicationBase.location.getCountry()!=null?" (" + ApplicationBase.location.getCountry() + ")":"");
    		}
    		
    		result = "'" + address + "'," 
    				+ ApplicationBase.location.getLocation().getLatitude() + "," 
    				+ ApplicationBase.location.getLocation().getLongitude()  + "," 
    				+ "'" + ApplicationBase.location.getCountryCode() + "'";
    		
    		//final String jsonResult = urlEncode(result);
        	final String jsonResult = result;
        	JSTaskExecutor.addTask(new Runnable() {				
				@Override
				public void run() {
					//We have to load in the webview thread
		        	String jsMethod = Constants.VV_JS_USERLOCATION_NOJSON_INFORM_METHOD; 
		        	if(ApplicationBase.enableLocationAfterInformLastLocation 
		        			&& isAndroidLastKnownLocation){
		        		jsMethod = Constants.VV_JS_USERLOCATION_LAST_KNOWN_NOJSON_INFORM_METHOD;
		        	}
		        	ToolBox.webview_runJavascript(pActivity.getWebView(), "javascript:" + jsMethod + "(" + jsonResult + ")", true, null);
					
					//Once informed, we stop the timer
					if(ApplicationBase.locationAcquireTimer!=null) {
			    		ApplicationBase.locationAcquireTimer.cancel();
			    		ApplicationBase.locationAcquireTimer = null;
			    	}
				}
			},"androidAppInformLocationToWeb_isLastLocKnown[" + isAndroidLastKnownLocation + "]");
        	
        	if(ApplicationBase.debugMode)
        		Log.d(Constants.TAG + "::WebJSInterface", "Sending current location [" + jsonResult + "]to the web, done.");        	
        	
    	}else{
    		//We bypass only if we wanted to set to NULL de location 
    		if(setToNone){
    			//We have to load in the webview thread
    			ToolBox.webview_runJavascript(pActivity.getWebView(), "javascript:" + Constants.VV_JS_USERLOCATION_CLEARLOCATION_METHOD + "()", true, null);    	        
            	if(ApplicationBase.debugMode)
            		Log.d(Constants.TAG + "::WebJSInterface", "Clearing current location in the web, done.");
    		}
    	}
    }	
    
    @JavascriptInterface
    public void androidAppInformLastLocationToWeb() {
    	if(ApplicationBase.appIsDestroyed)
    		return;
    	
    	//We only do this if we are in a page with location enabled
    	if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
    		//To enable for all versions we should add a method in the 
    		//page that tell to the page for the location.
	    	if(!ApplicationBase.isPageWithLocationBehavior)
	    		return;
    	}else{
    		//TODO In older versions we should avoid in other way
    	}
    	
    	//setAndroidAppLocation(address, lat, lng, country)
    	String result = "null,null,null,null";
    	
    	if(ApplicationBase.location!=null || ApplicationBase.locationLast!=null) {
    		LocationInfo location = (ApplicationBase.location!=null?ApplicationBase.location:ApplicationBase.locationLast);
    		String address = null;
    		if(location.getAddress()==null || 
    			(location.getAddress()!=null && location.getAddress().length()==0)){
    			//In case Android could not determine the address, we return an string.
    			address = mContext.getResources().getString(R.string.location_address_not_available);
    		}else{
    			address = ApplicationBase.location.getAddress() +
    					(ApplicationBase.location.getCity()!=null?" " + ApplicationBase.location.getCity():"") +
    					(ApplicationBase.location.getCountry()!=null?" (" + ApplicationBase.location.getCountry() + ")":"");
    		}
    		
    		result = "'" + address + "'," 
    				+ location.getLocation().getLatitude() + "," 
    				+ location.getLocation().getLongitude()  + "," 
    				+ "'" + location.getCountryCode() + "'";
    		
    		//final String jsonResult = urlEncode(result);
        	final String jsonResult = result;
        	//We have to load in the webview thread
        	ToolBox.webview_runJavascript(pActivity.getWebView(), "javascript:" + Constants.VV_JS_USERLOCATION_NOJSON_INFORM_METHOD + "(" + jsonResult + ")", true, null);
        	if(ApplicationBase.debugMode)
        		Log.d(Constants.TAG + "::WebJSInterface", "Sending current location [" + jsonResult + "]to the web, done.");        	
    	}
    }
    
    @JavascriptInterface
    public void androidCallWebGoBack() {
    	ToolBox.webview_runJavascript(pActivity.getWebView(), "javascript:" + Constants.VV_JS_WEB_GO_BACK_METHOD + "()", true, null);
    	if(ApplicationBase.debugMode)
    		Log.d(Constants.TAG + "::WebJSInterface", "Sending go back command to the web, done.");    	
    }
    
    @JavascriptInterface
    public void androidInformAppVersion() {
    	ToolBox.webview_runJavascript(pActivity.getWebView(), "javascript:" + Constants.VV_JS_WEB_INFORM_VERSION + "('" + ApplicationBase.versionString + "')", true, null);
    	if(ApplicationBase.debugMode)
    		Log.d(Constants.TAG + "::WebJSInterface", "Sending app version to the web, done.");    	
    }
    
    @JavascriptInterface
    public void androidIsLocationEnabled() {
    	if(ApplicationBase.debugMode)
    		Log.d(Constants.TAG + "::WebJSInterface", "Is Location Enabled asked from web.");
    	
		//If location is not enabled, we reset it in the web.
    	pActivity.getWebView().post(new Runnable() {
            public void run() {
            	if(!ToolBox.isLocationEnabled(mContext) || 
            			!ApplicationBase.permissionsLocationGranted ||
            			!ApplicationBase.enableLocation || 
                    	!ApplicationBase.enableInitialLocation){
            		ToolBox.webview_runJavascript(pActivity.getWebView(), "javascript:" + Constants.VV_JS_USERLOCATION_CLEARLOCATION_METHOD + "()", false, null);            		
            	}
            }
        });    	
    }
    
    @JavascriptInterface
    public void androidAppGetUserLocation() {
    	if(ApplicationBase.debugMode)
    		Log.d(Constants.TAG + "::WebJSInterface", "Current location command send from web.");
    	
    	String result = null;
        if(!ApplicationBase.enableLocation) {
        	if(ApplicationBase.debugMode)
        		Log.d(Constants.TAG, "Location module not enabled.");
        	//Latitude and longitude goes between -90 and 90.
        	result = "{\"resultCode\":\"module_not_enabled\",\"lat\":" + 100 + ",\"lng\":" + 100 + "}";            	
        }else{
        	if(ApplicationBase.locationServiceEnabled && ApplicationBase.location != null) {
        		result = "{\"resultCode\":\"module_enabled\",\"lat\":" + 
        					ApplicationBase.location.getLocation().getLatitude() + ",\"lng\":" 
        					+ ApplicationBase.location.getLocation().getLongitude() + "}";
        	}else{
        		//Latitude and longitude goes between -90 and 90.
                result = "{\"resultCode\":\"module_enabled\",\"lat\":" + 100 + ",\"lng\":" + 100 + "}";                
        	}
            
        	final String jsonResult = urlEncode(result);
        	//We have to load in the webview thread
        	ToolBox.webview_runJavascript(pActivity.getWebView(), "javascript:" + Constants.VV_JS_USERLOCATION_INFORM_METHOD + "('" + jsonResult + "')", true, null);        	            
        }
    }
    
    @JavascriptInterface
    public void androidAppRequestGPSLocation() {
    	if(ApplicationBase.debugMode)
    		Log.d(Constants.TAG + "::WebJSInterface", "Request GPS location command send from web.");
    	
    	if(!ApplicationBase.enableLocation) {
    		if(ApplicationBase.debugMode)
    			Log.d(Constants.TAG, "Location module not enabled.");
        	final String jsonResult = urlEncode("{\"resultCode\":\"module_not_enabled\",\"gpsEnabled\":false}");
        	//We have to load in the webview thread
        	ToolBox.webview_runJavascript(pActivity.getWebView(), "javascript:" + Constants.VV_JS_USERLOCATION_ENABLED_STATUS_METHOD + "('" + jsonResult + "')", true, null);
    	}else{
    		if(!ApplicationBase.locationServiceEnabled) {
    			if(ApplicationBase.debugMode)
    				Log.d(Constants.TAG, "Location module enabled but service not running.");
            	final String jsonResult = urlEncode("{\"resultCode\":\"service_not_running\",\"gpsEnabled\":false}");
            	//We have to load in the webview thread
            	ToolBox.webview_runJavascript(pActivity.getWebView(), "javascript:" + Constants.VV_JS_USERLOCATION_ENABLED_STATUS_METHOD + "('" + jsonResult + "')", true, null);            	
    		}else{
    			//Checks if the GPS is enabled, if not, it shows the 
            	//popup to enable it in the system settings.
            	if (!ApplicationBase.isGPSProviderEnabled) {
            		ToolBox.dialog_showGPSDisabledAlert(pActivity, 
            				"El GPS esta deshabilitado. Se necesita dicha función. ¿Desea hacerlo?", 
            				"Activar GPS", "Cancelar");        		
            	}else{
            		final String jsonResult = urlEncode("{\"resultCode\":\"module_enabled\",\"gpsEnabled\":true}");
            		//We have to load in the webview thread
            		ToolBox.webview_runJavascript(pActivity.getWebView(), "javascript:" + Constants.VV_JS_USERLOCATION_ENABLED_STATUS_METHOD + "('" + jsonResult + "')", true, null);            		
            	}
    		}
        }
    } 
    
    @JavascriptInterface
    public void androidLocationAcquired() {
    	if(ApplicationBase.debugMode)
    		Log.d(Constants.TAG, "Location successfully acquired from web.");
    	
    	pActivity.getWebView().post(new Runnable() {
            public void run() {
            	if(!ApplicationBase.appIsDestroyed)
            		//We stop the location service for now.
            		((ApplicationBase)pActivity.getApplication()).stopLocationService();
            }
        });
    }
    
    @JavascriptInterface
    public void androidRefresh() {
    	if(ApplicationBase.debugMode)
    		Log.d(Constants.TAG, "Refresh command send from web.");
    	
    	pActivity.getWebView().post(new Runnable() {
            public void run() {
            	pActivity.getWebView().reload();
            }
        });
    }
    
    @JavascriptInterface
    public void androidAppExit() {
    	if(ApplicationBase.debugMode)
    		Log.d(Constants.TAG, "Exit command send from web. Finishing app.");
    	
    	pActivity.finish();
    }                
    
    @JavascriptInterface
    public void getContent(String htmlRaw) {
    	//Usage:
    	//In onPageFinished:
    	//	view.loadUrl("javascript:window.Android.getContent('<html>' + escape(document.getElementsByTagName('html')[0].innerHTML) + '</html>');");
    	String htmlCode = null;
    	try{
    		htmlCode = URLDecoder.decode(htmlRaw, "UTF-8");
    	}catch(Exception e){}
    	if(ApplicationBase.debugMode)
    		Log.d(Constants.TAG, "Content: " + (htmlCode!=null?htmlCode:"not_available"));
    	
    	//We check for errors when loading.
    	//We do by checking HTML code because until Android M, itis not possible to get without 
    	//doing an additional call to the URL. The previous Android versions have the method 
    	// "onReceivedError" but it is only for network errors, no HTTP errors. In Android M
    	//we have "onReceivedHttpError()".
    	if(htmlCode!=null && 
    			(htmlCode.contains("Proxy Error") 
    			|| htmlCode.contains("Bad Gateway") 
    			|| htmlCode.contains("Service Unavailable"))
    		){
    		//We show something else than the default error page
    		ToolBox.application_runOnUIThread(null, new Runnable() {
				@Override
				public void run() {
					ApplicationBase.urlLoadingError = true;
					pActivity.showRefreshZoneErrorPage();
					pActivity.getWebView().stopLoading();
				}
			});
    	}else{
    		ApplicationBase.urlLoadingError = false;
    	}
    }
    
    @JavascriptInterface
    public void androidInformExternalId(Integer externalId) {
    	if(ApplicationBase.debugMode)
    		Log.d(Constants.TAG + "::WebJSInterface", "ExternalId received: " + externalId);
    	
    	ToucanClient.getInstance(mContext,
    			Constants.TOUCAN_API_TOKEN, 
    			Constants.TOUCAN_APP_PUB_KEY)
    			.informExternalId(externalId, null);
    }
    
    @JavascriptInterface
    public void androidInformProximityAPIKey(String proximityApiKey) {
    	if(proximityApiKey!=null && proximityApiKey.length()>0) {
    		ApplicationBase.EXTERNAL_PROXIMITY_API_TOKEN = proximityApiKey;
    		ToolBox.prefs_savePreference(mContext, ProximityExternalApiTask.PREF_FILE_NAME, ProximityExternalApiTask.PREF_PARAM_API_TOKEN, String.class, proximityApiKey);    		
    	}
    }
    
    //AUXILIAR
    
    
    /**
     * Returning a string implies to url encode
     * to avoid issues in the JS function when
     * using as function parameter.
     * 
     * @param data	data to url encode
     * @return  The url encoded result or ERROR string.
     */
    private String urlEncode(String data) {
    	String res = "ERROR";
		try {
			res = URLEncoder.encode(data, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			Log.e(Constants.TAG, "WebAppInterface. Error encoding to UTF-8 the result for the web [" + e.getMessage() + "]");
		}
		
		return res;
    }
    
    /**
     * To encode a part of a text. Use decodeWebComponent 
     * in JS in the web part.
     * 
     * @param data
     * @return
     */
    private String urlEncodeAddress(String data) {
    	String res = "ERROR";
		try {
			res = URLEncoder.encode(data.replaceAll("\\s+","_space_"), "UTF-8").replaceAll("_space_","%20");
		} catch (UnsupportedEncodingException e) {
			Log.e(Constants.TAG, "WebAppInterface. Error encoding to UTF-8 the result for the web [" + e.getMessage() + "]");
		}
		
		return res;
    }
}
