package com.colectivosvip.schindler.service.proximity;

import java.util.Calendar;
import java.util.Date;

import android.annotation.SuppressLint;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.Bundle;
import android.os.IBinder;
import android.util.Log;
import es.javocsoft.android.lib.toolbox.ToolBox;
import es.javocsoft.android.lib.toolbox.ToolBox.ApiLevel;
import es.javocsoft.android.lib.toolbox.ToolBox.LocationInfo;
import com.colectivosvip.schindler.ApplicationBase;

/**
 * A service that runs in background to watch for location changes in order to show, 
 * after some checks afterwards, notification alerts when offers are near to user
 * location. 
 * <br><br>
 * This service informs to an application about these events:
 * <ul>
 * 	 <li><b>LOCATION_PROXIMITY_SERVICE_STARTED</b>. Intent filter name: <i>com.colectivosvip.schindler.service.proximity.intent.action.LOCATION_PROXIMITY_SERVICE_STARTED</i></li>
 *   <li><b>LOCATION_PROXIMITY_SERVICE_SHUTDOWN</b>. Intent filter name: <i>com.colectivosvip.schindler.service.proximity.intent.action.LOCATION_PROXIMITY_SERVICE_SHUTDOWN</i></li>
 *   <li><b>LOCATION_PROXIMITY_CHANGED</b>. Intent filter name: <i>com.colectivosvip.schindler.service.proximity.intent.action.LOCATION_PROXIMITY_CHANGED</i></li>
 *   <li><b>LOCATION_PROXIMITY_GPS_ENABLED</b>. Intent filter name: <i>com.colectivosvip.schindler.service.proximity.intent.action.LOCATION_PROXIMITY_GPS_ENABLED</i></li>
 *   <li><b>LOCATION_PROXIMITY_GPS_DISABLED</b>. Intent filter name: <i>com.colectivosvip.schindler.service.proximity.intent.action.LOCATION_PROXIMITY_GPS_DISABLED</i></li>
 * </ul>
 * <br><br>
 * 
 * When a location change happens, a broadcast is sent, this broadcast will contain in its bundle:<br>
 * <ul>
 * <li>A {@link Location} in the bundle under the key LOCATION_KEY</li>
 * <li>A set of extra location information:
 * <ul>
 * 	<li>The country, under the key LOCATION_COUNTRY_KEY</li>
 *  <li>The country code, under the key LOCATION_COUNTRY_CODE_KEY</li>
 *  <li>The city, under the key LOCATION_CITY_KEY</li>
 *  <li>The address line, under the key LOCATION_ADDRESS_KEY</li>
 *  <li>The postal code, under the key LOCATION_POSTAL_CODE_KEY</li>
 * </ul>
 * </li>
 * </ul>
 * 
 * Declare the localization service in your AndroidManifest.xml:<br>
 * <code>
 * 	&lt;service android:name="com.colectivosvip.schindler.service.proximity.LocationProximityService" 
 *	     	 android:label="Proximity Location Service"
 *	     	 android:enabled="true"/&gt;
 * </code><br><br>
 * 
 * And implement in your application a receiver that listens for the desired events in order to 
 * react to them properly.
 * <br>
 * <code>
 * &lt;receiver android:name="your_application_receiver_package.LocationReceiver"
 *		    	  android:enabled="true" 
 *		    	  android:exported="false"/&gt;
 *		    &lt;intent-filter/&gt;
 *               &lt;action android:name="com.colectivosvip.schindler.service.proximity.intent.action.LOCATION_PROXIMITY_SERVICE_STARTED" /&gt;
 *               &lt;action android:name="com.colectivosvip.schindler.service.proximity.intent.action.LOCATION_PROXIMITY_SERVICE_SHUTDOWN" /&gt;
 *               &lt;action android:name="com.colectivosvip.schindler.service.proximity.intent.action.LOCATION_PROXIMITY_CHANGED" /&gt;
 *               &lt;action android:name="com.colectivosvip.schindler.service.proximity.intent.action.LOCATION_PROXIMITY_GPS_ENABLED" /&gt;
 *               &lt;action android:name="com.colectivosvip.schindler.service.proximity.intent.action.LOCATION_PROXIMITY_GPS_DISABLED" /&gt;
 *           &lt;/intent-filter/&gt;            
 *	&lt;/receiver/&gt;
 * </code>
 * 
 * <br><br>
 * <b>Notes</b><br>
 * <ul>
 * <li>The localization service can be customized when starting by setting:
 *   <ul>
 *   	<li>Distance (in meters) between localization changes. Default is 100 meters.</li>
 * 		<li>Time between localization changes. Default is 60 seconds (60000 milliseconds).</li>
 * 		<li>Time between alerts. Default is 10 minutes. This avoids spam.</li>
 * 		<li>Accuracy change threshold (in meters). Default is 5 meters.</li>
 * 		<li>Use also the GPS, if available, or just the Wi-Fi and Radio signals. If not set GPS is not used.</li>
 * 		<li>The algorithm type (an String) to determine if a new location is better than 
 * 			than the previous one. We can choose between SIMPLE or COMPLEX, 
 * 			see {@link LOCATION_ALGORITHM_TYPE}. If not set, SIMPLE is used.
 * 			<ul>
 * 				<li>SIMPLE. Choose this if we want only to be alerted every time me move the
 * 					specified distance in the parameter LOCATION_SERVICE_PARAM_MIN_DISTANCE.</li>
 * 				<li>COMPLEX. Choose this when we want also to check the accuracy and the age 
 * 					of locations and not only the distance.</li>
 * 			</ul>
 * 		</li>
 * 		<li>We can tell to the service if the service is being started after a device reboot. We should provide this information in order to be able to get the correct stored parameters.</li>
 *   </ul>  
 *   To set these values, set them through the service starting intent by 
 *   using these keys in the bundle:
 *   <ul>
 *     <li>LOCATION_SERVICE_PARAM_MIN_DISTANCE</li> 
 * 	   <li>LOCATION_SERVICE_PARAM_MIN_TIME</li>
 * 	   <li>LOCATION_PROXIMITY_UPDATE_MIN_TIME_BETWEEN_ALERTS</li>
 * 	   <li>LOCATION_SERVICE_PARAM_ACCURACY_THRESHOLD</li>
 * 	   <li>LOCATION_SERVICE_PARAM_USE_GPS</li>
 *     <li>LOCATION_SERVICE_PARAM_ALGORITHM</li>
 *     <li>LOCATION_SERVICE_PARAM_DEVICE_BOOTED</li> 	   
 *   </ul>
 *  </li> 
 *  <li>If service gets stopped, it will automatically run again.</li>
 * </ul>
 * 
 * <br>See:<br><br>
 * Services<br>
 *  http://developer.android.com/intl/es/reference/android/app/Service.html<br>
 *  http://www.vogella.com/tutorials/AndroidServices/article.html<br>
 *  http://developer.android.com/intl/es/training/run-background-service/send-request.html<br><br>
 *  
 *  
 * Location<br>
 * 	http://developer.android.com/guide/topics/location/strategies.html<br>
 *  http://developer.android.com/intl/es/reference/android/location/LocationManager.html<br>
 *  http://developer.android.com/intl/es/reference/android/location/LocationListener.html<br>
 * 
 * 
 * @author JavocSoft 2016
 * @version 1.0<br>
 * $Rev: 875 $<br>
 * $LastChangedDate: 2016-12-15 18:50:05 +0100 (jue, 15 dic 2016) $<br>
 * $LastChangedBy: jgonzalez $
 *
 */
public class LocationProximityService extends Service implements LocationListener {

	private static final String TAG = "Location Proximity Service";
	public static final String PREF_FILE_NAME = "prefs_proximity_location";
	
	public static final String LOCATION_SERVICE_PARAM_MIN_DISTANCE = "LOCATION_PROXIMITY_UPDATE_MIN_DISTANCE";
	public static final String LOCATION_SERVICE_PARAM_MIN_TIME = "LOCATION_PROXIMITY_UPDATE_MIN_TIME";
	public static final String LOCATION_SERVICE_PARAM_ACCURACY_THRESHOLD = "LOCATION_PROXIMITY_UPDATE_ACCURACY_THRESHOLD";
	public static final String LOCATION_SERVICE_PARAM_MIN_TIME_BETWEEN_ALERTS = "LOCATION_PROXIMITY_UPDATE_MIN_TIME_BETWEEN_ALERTS";
	public static final String LOCATION_SERVICE_PARAM_USE_GPS = "LOCATION_PROXIMITY_USE_GPS";
	public static final String LOCATION_SERVICE_PARAM_ALGORITHM =  "LOCATION_PROXIMITY_ALGORITHM";
	public static final String LOCATION_SERVICE_PARAM_DEVICE_BOOTED =  "LOCATION_PROXIMITY_DEVICE_BOOTED";
	
	public static final String LOCATION_SERVICE_PARAM_LATITUDE =  "LOCATION_PROXIMITY_LATITUDE";
	public static final String LOCATION_SERVICE_PARAM_LONGITUDE =  "LOCATION_PROXIMITY_LONGITUDE";
	public static final String LOCATION_SERVICE_PARAM_TIME =  "LOCATION_PROXIMITY_TIME";
	public static final String LOCATION_SERVICE_PARAM_ACCURACY =  "LOCATION_PROXIMITY_ACCURACY";
	public static final String LOCATION_SERVICE_PARAM_PROVIDER =  "LOCATION_PROXIMITY_PROVIDER";
	public static final String LOCATION_SERVICE_PARAM_ALTITUDE =  "LOCATION_PROXIMITY_ALTITUDE";
	public static final String LOCATION_SERVICE_PARAM_BEARING =  "LOCATION_PROXIMITY_BEARING";
	public static final String LOCATION_SERVICE_PARAM_SPEED =  "LOCATION_PROXIMITY_SPEED";
	public static final String LOCATION_SERVICE_PARAM_REALTIMENANOS =  "LOCATION_PROXIMITY_REALTIMENANOS";
	
	
	public static final String ACTION_LOCATION_SERVICE_STARTED = LocationProximityService.class.getPackage().getName() + ".intent.action.LOCATION_PROXIMITY_SERVICE_STARTED";
	public static final String ACTION_LOCATION_SERVICE_SHUTDOWN = LocationProximityService.class.getPackage().getName() + ".intent.action.LOCATION_PROXIMITY_SERVICE_SHUTDOWN";
	public static final String ACTION_LOCATION_CHANGED = LocationProximityService.class.getPackage().getName() + ".intent.action.LOCATION_PROXIMITY_CHANGED";
	public static final String ACTION_LOCATION_GPS_ENABLED = LocationProximityService.class.getPackage().getName() + ".intent.action.LOCATION_PROXIMITY_GPS_ENABLED";
	public static final String ACTION_LOCATION_GPS_DISABLED = LocationProximityService.class.getPackage().getName() + ".intent.action.LOCATION_PROXIMITY_GPS_DISABLED";
	
	public static final String LOCATION_KEY = "location";
	public static final String LOCATION_COUNTRY_KEY = "location_country";
	public static final String LOCATION_COUNTRY_CODE_KEY = "location_country_code";
	public static final String LOCATION_CITY_KEY = "location_city";
	public static final String LOCATION_ADDRESS_KEY = "location_address";
	public static final String LOCATION_POSTAL_CODE_KEY = "location,postal_code";
	
	private static final int TWO_MINUTES = (1000*60)*2;
	
	private static boolean svcStarted = false;
	
	
	/**
     * The algorithm to use to determine if the new position should rise a new location
     * alert broadcast intent.<br>
     * <ul>
     * 	<li>SIMPLE. Choose this if we want only to be alerted every time me move the
     * 		specified distance in the parameter LOCATION_SERVICE_PARAM_MIN_DISTANCE.</li>
     * 	<li>COMPLEX. Choose this when we want also to check the accuracy, provider and the age 
     * 		of locations and not only the distance.</li>
     * </ul>		
     */
    public static enum LOCATION_ALGORITHM_TYPE {SIMPLE, COMPLEX};
	
    private static final int UPDATE_MIN_DISTANCE = 500; //meters 
    //Milliseconds. This minimum time of 45 seconds is harcoded in Android
    private static final long UPDATE_MIN_TIME = 10*(60 * 1000); //Milliseconds
    private static final int UPDATE_ACCURACY_THRESHOLD = 5; //meters
    private static final long UPDATE_MIN_TIME_BEWTEEN_ALERTS = 10*(60 * 1000);
    private static final LOCATION_ALGORITHM_TYPE LOCATION_ALGORITHM = LOCATION_ALGORITHM_TYPE.SIMPLE;
        
    public LocationManager locationManager;
    public Location previousBestLocation = null;
    private int minDistance = UPDATE_MIN_DISTANCE;
    private long minTime = UPDATE_MIN_TIME;
    private int accuracyThreshold = UPDATE_ACCURACY_THRESHOLD;
    private boolean useGPS = false;
    private LOCATION_ALGORITHM_TYPE locAlgorithm = LOCATION_ALGORITHM;
    private long minTimeBetweenAlerts = UPDATE_MIN_TIME_BEWTEEN_ALERTS;
    
	
	public LocationProximityService() {}

	
	@Override
    public void onCreate() {
		super.onCreate();
		if(!svcStarted)
			Log.d(TAG, "Location proximity service created.");
    }
	
	@Override
	public IBinder onBind(Intent intent) {
		return null;
	}
	
	@Override
	public int onStartCommand(Intent intent, int flags, int startId) {
		
		if ((flags & START_FLAG_REDELIVERY)!=0) {
			//Service was restarted.
			if(ApplicationBase.debugMode)
	        	Log.d(TAG, "Location proximity service re-started.");			
		}
		
		//We only do this in case we have not yet started the service.
		if(!svcStarted) {
			Log.d(TAG, "Location proximity service starting...");			
			doOnStart(intent);
			svcStarted = true;
		}
		
		//START_STICKY, START_NOT_STICKY and START_REDELIVER_INTENT are 
		//only relevant when the phone runs out of memory and kills the 
		//service before it finishes executing. Also if the user kills the
		//application from tasks menu. See:
		//
		// http://android-developers.blogspot.com.au/2010/02/service-api-changes-starting-with.html
		//
		// - START_STICKY tells the OS to recreate the service after it 
		//	 has enough memory and call onStartCommand() again with a 
		//	 null intent. 
		// - START_NOT_STICKY tells the OS to not bother recreating the 
		//   service again. 
		// - START_REDELIVER_INTENT that tells the OS to recreate the 
		//   service AND re-deliver the same intent to onStartCommand().
		return Service.START_REDELIVER_INTENT;
	}
	
    @Override
    public void onDestroy() {       
    	Log.d(TAG, "Location proximity service destroyed");
        super.onDestroy();
        
        previousBestLocation = null;
        locationManager.removeUpdates(this);
        locationManager = null;
        
        svcStarted = false;
        
        deliverBroadcast(ACTION_LOCATION_SERVICE_SHUTDOWN, null);
    }
    
    
    //AUXILIAR
    
    private void doOnStart(Intent intent) {
    	
    	if(intent!=null && !intent.hasExtra(LOCATION_SERVICE_PARAM_DEVICE_BOOTED)) {
    		//Note: After booting the service is restarted but the intent has no data in this case.
    		
    		//We get the initialization parameters from the intent.
    		minDistance = intent.getIntExtra(LOCATION_SERVICE_PARAM_MIN_DISTANCE, UPDATE_MIN_DISTANCE);
        	minTime = intent.getLongExtra(LOCATION_SERVICE_PARAM_MIN_TIME, UPDATE_MIN_TIME);
        	accuracyThreshold = intent.getIntExtra(LOCATION_SERVICE_PARAM_ACCURACY_THRESHOLD, UPDATE_ACCURACY_THRESHOLD);
        	useGPS = intent.getBooleanExtra(LOCATION_SERVICE_PARAM_USE_GPS, false);
        	locAlgorithm = getLocationAlgorithmFromString(intent.getStringExtra(LOCATION_SERVICE_PARAM_ALGORITHM));
        	minTimeBetweenAlerts = intent.getLongExtra(LOCATION_SERVICE_PARAM_MIN_TIME_BETWEEN_ALERTS, UPDATE_MIN_TIME_BEWTEEN_ALERTS);
        	
        	//We save the initialization for later usage in case services gets rebooted.
        	Thread t = new Thread(new Runnable() {				
				@Override
				public void run() {
					ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_MIN_DISTANCE, Integer.class, minDistance);
		        	ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_MIN_TIME, Long.class, minTime);
		        	ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_MIN_TIME_BETWEEN_ALERTS, Long.class, minTimeBetweenAlerts);		        	
		        	ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_ACCURACY_THRESHOLD, Integer.class, accuracyThreshold);
		        	ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_USE_GPS, Boolean.class, useGPS);
		        	ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_ALGORITHM, String.class, locAlgorithm.name());
				}
			});
        	t.start();
        	
    	}else{
    		if(ApplicationBase.debugMode) {
    			if(intent.hasExtra(LOCATION_SERVICE_PARAM_DEVICE_BOOTED)){    			
        			Log.d(TAG, "Location proximity service starting after a device reboot.");
        		}
    		}
    		
    		//No data in the intent or the service is started after a device reboot, 
    		//we try to get from saved preferences if there is one.
    		if(ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_MIN_DISTANCE)) {
    			minDistance = ((Integer)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_MIN_DISTANCE, Integer.class)).intValue();
    		}
    		if(ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_MIN_TIME)) {
    			minTime = ((Long)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_MIN_TIME, Long.class)).longValue();
    		}
    		if(ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_MIN_TIME_BETWEEN_ALERTS)) {
    			minTimeBetweenAlerts = ((Long)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_MIN_TIME_BETWEEN_ALERTS, Long.class)).longValue();
    		}
    		if(ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_ACCURACY_THRESHOLD)) {
    			accuracyThreshold = ((Integer)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_ACCURACY_THRESHOLD, Integer.class)).intValue();
    		}
    		if(ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_USE_GPS)) {
    			useGPS = ((Boolean)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_USE_GPS, Boolean.class));
    		}
    		if(ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_ALGORITHM)) {
    			String locAlg = ((String)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_ALGORITHM, String.class));
    			locAlgorithm = getLocationAlgorithmFromString(locAlg);
    		}
    	}
    	
    	if(ToolBox.permission_areGranted(getBaseContext(), ToolBox.PERMISSION_LOCATION.keySet())) {
    		locationManager = (LocationManager) getSystemService(Context.LOCATION_SERVICE);
    		locationManager.requestLocationUpdates(
            		LocationManager.NETWORK_PROVIDER, 
            		minTime, minDistance, this);
    		if(useGPS) {
    			locationManager.requestLocationUpdates(
            		LocationManager.GPS_PROVIDER, 
            		minTime, minDistance, this);
    		}
	        
	        if(ApplicationBase.debugMode)
	        	Log.d(TAG, "Location proximity service started. Parameters 'minTime': " + minTime + " / 'minDistance': " + minDistance + " / 'accuracyUmbral': " + accuracyThreshold + " / 'useGPS': " + useGPS + " / 'Location Algorithm': " + locAlgorithm.name());	            
	        
	        deliverBroadcast(ACTION_LOCATION_SERVICE_STARTED, null);
    	}else{
    		Log.d(TAG, "Location proximity service not started, permissions not granted. Parameters 'minTime': " + minTime + " / 'minDistance': " + minDistance + " / 'accuracyUmbral': " + accuracyThreshold + " / 'useGPS': " + useGPS + " / 'Location Algorithm': " + locAlgorithm.name());
    	}
    }
	
    
    protected LOCATION_ALGORITHM_TYPE getLocationAlgorithmFromString(String locationAlgorithm) {
    	LOCATION_ALGORITHM_TYPE res = null;
    	try{
    		res = LOCATION_ALGORITHM_TYPE.valueOf(locationAlgorithm);
    	}catch(Exception e){
    		//We use the SIMPLE mode and log the error.
    		res = LOCATION_ALGORITHM_TYPE.SIMPLE;
    		if(ApplicationBase.debugMode)
    			Log.d(TAG, "Location Service: could not determine the location algorithm type [" + locationAlgorithm + "]. Using SIMPLE by default.");
    	}
    	
    	return res;
    }
    
    protected boolean isBetterLocation(Location location, Location currentBestLocation) {
    	boolean res = false;
    	
    	if(location.getLatitude()!=0 && location.getLongitude()!=0){ //Only valid locations
    		switch (locAlgorithm) {
				case SIMPLE:
					res = isBetterLocationSimple(location, currentBestLocation);
					break;
				case COMPLEX:			
					res = isBetterLocationComplex(location, currentBestLocation);			
			}
    	}else{
    		if(ApplicationBase.debugMode)
        		Log.d(TAG, "Proximity service: isBetterLocation. Location is 0,0");
    	}
    	
    	return res;
    }
    
    /**
     * Checks if the new location is better than the old one.
     * 
     * @param location
     * @param currentBestLocation
     * @return
     */
	protected boolean isBetterLocationComplex(Location location, Location currentBestLocation) {
        if (currentBestLocation == null) {
            return true; //New location is better than no location
        }

        //Check if new location is newer or older
        long timeDelta = location.getTime() - currentBestLocation.getTime();
        boolean isSignificantlyNewer = timeDelta > TWO_MINUTES;
        boolean isSignificantlyOlder = timeDelta < -TWO_MINUTES;
        boolean isNewer = timeDelta > 0;
        
        if(currentBestLocation.getLatitude()==location.getLatitude() &&
           currentBestLocation.getLongitude()==location.getLongitude()) {
        	//Same location, new measurement is newer yes but the location is the same.
        	isNewer = false;
        	if(ApplicationBase.debugMode)
        		Log.d(TAG, "Proximity service (ALG: COMPLEX): isNever set to FALSE (same location).");
        }else{
        	double distanceBetweenMeasurements = ToolBox.location_distance(location.getLatitude(), location.getLongitude(), 
        			currentBestLocation.getLatitude(), currentBestLocation.getLongitude());
        	if(ApplicationBase.debugMode)
        		Log.d(TAG, "Proximity service (ALG: COMPLEX): Elapsed distance (Haversine) since last location: " + distanceBetweenMeasurements);
        	
        	if(distanceBetweenMeasurements<=minDistance){
        		//New measurement is newer yes but the distance between last and new location is less
        		//than the minimal distance.
        		isNewer = false;
        		if(ApplicationBase.debugMode)
            		Log.d(TAG, "Proximity service (ALG: COMPLEX): isNever set to FALSE (Haversine distance less than threshold).");
        	}
        }

        //If it's been more than two minutes since the current location, we use the 
        //new location because the user has probably moved.
        if (isSignificantlyNewer) {
            return true;
        } else if (isSignificantlyOlder) {
        	if(ApplicationBase.debugMode)
        		Log.d(TAG, "Proximity service (ALG: COMPLEX): Location is not better: isSignificantlyOlder");
            return false; //If the new location older than two minutes, should be worse
        }

        //Check if the new location is more or less accurate
        int accuracyDelta = (int) (location.getAccuracy() - currentBestLocation.getAccuracy());
        boolean isLessAccurate = accuracyDelta > 0;
        boolean isMoreAccurate = false; 
        if(accuracyDelta < 0 && ((-1)*accuracyDelta)>=accuracyThreshold) {
        	isMoreAccurate = true;
        }
        boolean isSignificantlyLessAccurate = accuracyDelta > 200;
        if(ApplicationBase.debugMode)
    		Log.d(TAG, "Proximity service (ALG: COMPLEX): isMore accurated? " + isMoreAccurate);

        //Check if the old and new location have the same provider
        boolean isFromSameProvider = 
        		isSameProvider(location.getProvider(), currentBestLocation.getProvider());

        //Watch for location quality using a combination of timeliness and accuracy
        if (isMoreAccurate) {
            return true;
        } else if (isNewer && !isLessAccurate) {
            return true;
        } else if (isNewer && !isSignificantlyLessAccurate && isFromSameProvider) {
            return true;
        }
        
        if(ApplicationBase.debugMode)
        	Log.d(TAG, "Proximity service (ALG: COMPLEX): is not better: isNewer: " + isNewer + 
        			", isLessAccurate: " + isLessAccurate + 
        			", isSignificantlyLessAccurate: " + isSignificantlyLessAccurate + 
        			", isFromSameProvider: " + isFromSameProvider);
        
        return false;
    }
	
	protected boolean isBetterLocationSimple(Location location, Location currentBestLocation) {
		boolean shouldAlert = true;
		
		if (currentBestLocation == null) {
			if(ApplicationBase.debugMode)
        		Log.d(TAG, "Proximity Service (ALG: SIMPLE): shouldAlert set to TRUE. currentBestLocation is NULL.");
            return true; //New location is better than no location
        }
        
        if(currentBestLocation.getLatitude()==location.getLatitude() &&
           currentBestLocation.getLongitude()==location.getLongitude()) {
        	//Same location, new measurement is newer yes but the location is the same.
        	shouldAlert = false;
        	if(ApplicationBase.debugMode)
        		Log.d(TAG, "Proximity Service (ALG: SIMPLE): shouldAlert set to FALSE. (same location).");
        }else{
        	double distanceBetweenMeasurements = ToolBox.location_distance(location.getLatitude(), location.getLongitude(), 
        			currentBestLocation.getLatitude(), currentBestLocation.getLongitude());
        	if(ApplicationBase.debugMode)
        		Log.d(TAG, "Proximity Service (ALG: SIMPLE): Elapsed distance (Haversine) since last location: " + distanceBetweenMeasurements + ", minDistance: " + minDistance);
        	
        	if(distanceBetweenMeasurements<minDistance){
        		//New measurement is newer yes but the distance between last and new location is less
        		//than the minimal distance.
        		shouldAlert = false;
        		if(ApplicationBase.debugMode)
            		Log.d(TAG, "Proximity Service (ALG: SIMPLE): shouldAlert set to FALSE (Haversine distance less than threshold). Distance: " + distanceBetweenMeasurements + ", minDistance: " + minDistance);        		
        	}else{
        		//See https://currentmillis.com/
        		Calendar bLocationTime = Calendar.getInstance();        		
        		bLocationTime.setTime(new Date(currentBestLocation.getTime()));
        		Calendar nextLocationAlert = Calendar.getInstance();
        		nextLocationAlert.setTimeInMillis(bLocationTime.getTimeInMillis());
        		nextLocationAlert.add(Calendar.MILLISECOND, Integer.parseInt(String.valueOf(minTimeBetweenAlerts)));
        		
        		if(location.getTime()<(nextLocationAlert.getTimeInMillis())){
        			//The new measure has the minimum distance but still the time between alerts
        			//is too short.
        			shouldAlert = false;
        			if(ApplicationBase.debugMode)
                		Log.d(TAG, "Proximity Service (ALG: SIMPLE): shouldAlert set to FALSE (Time between alerts less than threshold). No alerts until [" + nextLocationAlert.getTime() + "]. Last alert was at: [" + bLocationTime.getTime() + "]. Received Location Time: [" + (new Date(location.getTime())) + "].");
        		}else{
        			if(ApplicationBase.debugMode)
                		Log.d(TAG, "Proximity Service (ALG: SIMPLE): shouldAlert set to TRUE (Haversine distance:" + distanceBetweenMeasurements + " (minDistance: " + minDistance + "), time between alerts is OK. Last alert was at: [" + bLocationTime.getTime() + "]. Received Location Time: [" + (new Date(location.getTime())) + "]. No alerts until: [" + nextLocationAlert.getTime() + "].");
        		}
        	}
        }
        
        return shouldAlert;
    }

	/** 
	 * Checks if two location providers are the same.
	 * */
    private boolean isSameProvider(String provider1, String provider2) {
        if (provider1 == null) {
          return provider2 == null;
        }
        return provider1.equals(provider2);
    }
    
    /**
     * An utility method to perform a job in a separated thread.
     */
    public static Thread doInBackgroundThread(final Runnable runnable) {
        final Thread t = new Thread() {
            @Override
            public void run() {
                try { runnable.run(); } finally {}
            }
        };
        t.start();
        return t;
    }
    
    
    
    // LocationListener methods ---------------------------------------------
    
    @SuppressLint("NewApi")
	public void onLocationChanged(final Location loc) {
      	if(ApplicationBase.debugMode)
       		Log.d(TAG, "Proximity Service: Location changed.");
      	
      	
      	if(previousBestLocation==null) {
      		previousBestLocation = recreateLocationObject();
      		//If after this point is still null, is the first time we get a location.
      	}
      	
      	if(isBetterLocation(loc, previousBestLocation)) {
      		//...some extra information about the location
	        LocationInfo locInfo = ToolBox.location_addressInfo(getBaseContext(), loc.getLatitude(), loc.getLongitude());
      		if(locInfo==null)
      			return;
      		
      		previousBestLocation = loc;
      		
      		//We save the parameters we need just in case service is killed for afterwards recreation of last location.
      		Thread t = new Thread(new Runnable() {	
				@Override
				public void run() {
					ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_LATITUDE, Double.class, previousBestLocation.getLatitude());
					ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_LONGITUDE, Double.class, previousBestLocation.getLongitude());
					ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_ACCURACY, Float.class, previousBestLocation.getAccuracy());
					ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_TIME, Long.class, previousBestLocation.getTime());
					ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_PROVIDER, String.class, previousBestLocation.getProvider());
					//Not used yet
					ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_ALTITUDE, Double.class, previousBestLocation.getAltitude());
					ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_BEARING, Float.class, previousBestLocation.getBearing());
					if(ToolBox.device_hasAPILevel(ApiLevel.LEVEL_17)){
						ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_REALTIMENANOS, Long.class, previousBestLocation.getElapsedRealtimeNanos());
					}
					ToolBox.prefs_savePreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_SPEED, Float.class, previousBestLocation.getSpeed());
				}
			});
        	t.start();
      			
	        //Send the change to an application receiver.
	        Bundle extras = new Bundle();
	        extras.putParcelable(LOCATION_KEY, loc);
	        //...some extra information about the location
	        extras.putString(LOCATION_COUNTRY_KEY, locInfo.getCountry());
	        extras.putString(LOCATION_COUNTRY_CODE_KEY, locInfo.getCountryCode());
	        extras.putString(LOCATION_CITY_KEY, locInfo.getCity());
	        extras.putString(LOCATION_ADDRESS_KEY, locInfo.getAddress());
	        extras.putString(LOCATION_POSTAL_CODE_KEY, locInfo.getPostalCode());
	        
	        deliverBroadcast(ACTION_LOCATION_CHANGED, extras);
	        if(ApplicationBase.debugMode)
	    		Log.d(TAG, "Proximity Service: Location change broadcast message sent.");
      	}                               
    }

    public void onProviderDisabled(String provider) {
      	if(ApplicationBase.debugMode)
       		Log.d(TAG, "Proximity Service: Location provider disabled [" + provider + "].");
      	if(provider.equals(LocationManager.GPS_PROVIDER)){
       		previousBestLocation = null;
       		deliverBroadcast(ACTION_LOCATION_GPS_DISABLED, null);
       	}
    }

    public void onProviderEnabled(String provider) {
      	if(ApplicationBase.debugMode)
      		Log.d(TAG, "Proximity Service: Location provider enabled [" + provider + "].");
       	if(provider.equals(LocationManager.GPS_PROVIDER)){
       		previousBestLocation = null;
       		deliverBroadcast(ACTION_LOCATION_GPS_ENABLED, null);        		
       	}
    }

    public void onStatusChanged(String provider, int status, Bundle extras) {}
    
    //End LocationListener methods ---------------------------------------------
    
    
    /**
     * Delivers a broadcast action.
     * 
     * @param action
     * @param extras
     */
    @SuppressLint("InlinedApi")
	private void deliverBroadcast(String action, Bundle extras){
    	Intent intent = new Intent(action);
    	
    	//Since Android 3.0, the intent would not be received by the broadcast receiver. 
    	//This is because Android 3.0 introduced a launch control security measure that 
    	//prevents components of stopped applications from being launched via an intent. 
    	//An application is considered to be in a stopped state if the application has 
    	//either just been installed and not previously launched, or been manually stopped 
    	//by the user using the application manager on the device. To get around this, 
    	//however, a flag can be added to the intent before it is sent to indicate that 
    	//the intent is to be allowed to start a component of a stopped application.
    	//
    	//This is an anti-malware move by Google. Google has advocated that users should 
    	//launch an activity from the launcher first, before that application can go do 
    	//much. Preventing BOOT_COMPLETED from being delivered until the activity is 
    	//launched is a logical consequence of the that argument.
    	if(ToolBox.device_hasAPILevel(ApiLevel.LEVEL_12)){
    		if(ApplicationBase.debugMode)
          		Log.d(TAG, "Proximity Service: Adding flag 'FLAG_INCLUDE_STOPPED_PACKAGES'.");
    		intent.addFlags(Intent.FLAG_INCLUDE_STOPPED_PACKAGES);
    	}
    	
    	if(extras!=null) {
    		intent.putExtras(extras);
    	}
    	sendBroadcast(intent);
    }
    
    /**
     * We recreate, not including the original extras of the Location object, the 
     * previous location object stored in shared preferences.
     */
    @SuppressLint("NewApi")
	private Location recreateLocationObject() {
    	Location res = null;
    	
    	if(ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_LATITUDE) && 
      		ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_LONGITUDE) &&
      		ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_ACCURACY) &&
      		ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_TIME) &&
      		ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_PROVIDER)){
      		
    		res = new Location((String)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_PROVIDER, String.class));
    		res.setLatitude((Double)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_LATITUDE, Double.class));
    		res.setLongitude((Double)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_LONGITUDE, Double.class));
    		res.setAccuracy((Float)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_ACCURACY, Float.class));
    		res.setTime((Long)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_TIME, Long.class));
    		
    		//Other not used yet fields
    		if(ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_ALTITUDE))
    			res.setAltitude((Double)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_ALTITUDE, Double.class));
    		if(ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_BEARING))
    			res.setBearing((Float)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_BEARING, Float.class));
    		if(ToolBox.device_hasAPILevel(ApiLevel.LEVEL_17) && ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_REALTIMENANOS)){
    			res.setElapsedRealtimeNanos((Long)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_REALTIMENANOS, Long.class));
    		}
    		if(ToolBox.prefs_existsPref(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_SPEED))
    			res.setSpeed((Float)ToolBox.prefs_readPreference(getBaseContext(), PREF_FILE_NAME, LOCATION_SERVICE_PARAM_SPEED, Float.class));
      	}
    	
    	return res;
    }
}
