package es.javocsoft.testing.app00.receiver;

import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.os.SystemClock;
import android.support.v4.content.WakefulBroadcastReceiver;
import android.util.Log;
import es.javocsoft.testing.app00.ApplicationBase;
import es.javocsoft.testing.app00.Constants;
import es.javocsoft.testing.app00.service.SimpleWakefulService;

/**
 * Use this receiver when you have to do a long time job. This type of receiver 
 * guarantees that CPU will stay awake until you fire completeWakefulIntent.
 * 
 * In this WakefulBroadcastReceiver you should do the job using the method 
 * startWakefulService() to start the service that does the work. This method 
 * is comparable to startService(), except that the WakefulBroadcastReceiver is 
 * holding a wake lock when the service starts. The intent that is passed with 
 * startWakefulService() holds an extra identifying the wake lock.<br><br>
 * 
 * See:<br> 
 * 	https://developer.android.com/intl/es/reference/android/support/v4/content/WakefulBroadcastReceiver.html
 * <br><br>
 * <b>Notes</b>:
 * <br><br>
 * WakefulBroadcastReceiver is a helper for the common pattern of implementing 
 * a BroadcastReceiver that receives a device wakeup event and then passes the 
 * work off to a Service, while ensuring that the device does not go back to 
 * sleep during the transition.
 * <br><br>
 * 
 * To use it, declare in your AndroidManifest.xml the receiver:<br><br>
 * 
 * &lt;receiver android:name="es.javocsoft.testing.app00.receiver.SimpleWakefulReceiver"<br>
 *	     	    android:enabled="true" <br>
 *		    	android:exported="false" &gt;<br>
 *		    &lt;intent-filter&gt;<br>
 *              &lt;action android:name="your.app.package.SOME_ACTION" /&gt;<br>                
 *          &lt;/intent-filter&gt;<br>
 * &lt;/receiver&gt;
 *		
 * 
 * @author JavocSoft 2013
 * @version 2.0<br>
 * $Rev: 816 $<br>
 * $LastChangedDate: 2015-11-30 15:03:36 +0100 (lun, 30 nov 2015) $<br>
 * $LastChangedBy: admin $
 *
 */
public class SimpleWakefulReceiver extends WakefulBroadcastReceiver {
	
	public static final String ACTION_1 = Constants.APP_RECEIVER_ACTION_PREFIX + 
				"intent.action.ACTION_1";
	
	public SimpleWakefulReceiver() {}

	@Override
	public void onReceive(Context context, Intent intent) {
		
		Log.d(Constants.TAG, "Application receiver.");
		   
		//Implement as required.
		Bundle bundle = intent.getExtras();		
		if(intent.getAction().equals(ACTION_1)) {   
			// This is the Intent to deliver to our service.
	        Intent service = new Intent(context, SimpleWakefulService.class);

	        // Start the service, keeping the device awake while it is launching.
	        if(ApplicationBase.debugMode)
	        	Log.d(Constants.TAG, "Starting Wakeful service @ " + SystemClock.elapsedRealtime());
	        startWakefulService(context, service);
		}else{
			//Not a valid action
			completeWakefulIntent(intent);
		}		
		
	}

}
