package com.colectivosvip.optimil.barcodescan;


import net.sourceforge.zbar.Config;
import net.sourceforge.zbar.Image;
import net.sourceforge.zbar.ImageScanner;
import net.sourceforge.zbar.Symbol;
import net.sourceforge.zbar.SymbolSet;
import android.app.Activity;
import android.content.Intent;
import android.hardware.Camera;
import android.hardware.Camera.AutoFocusCallback;
import android.hardware.Camera.PreviewCallback;
import android.hardware.Camera.Size;
import android.media.MediaPlayer;
import android.media.MediaPlayer.OnCompletionListener;
import android.os.Bundle;
import android.os.Handler;
import android.widget.FrameLayout;
import es.javocsoft.android.lib.toolbox.ToolBox;
import com.colectivosvip.optimil.R;


/**
 * Activity that uses the ZXING core library JAR to scan Scans a Barcode/QR.
 * <br><br>
 * Internally it uses the ZBAR native Android library. This library must be re-compiled
 * in order to work with Android 6+, see https://github.com/dm77/barcodescanner#rebuilding-zbar-libraries
 * 
 * This activity must be declared in AndroidManifest.xml and also include the camera permission.
 * 
 * @author JavocSoft 2016<br>
 * $Rev: 871 $<br>
 * $LastChangedDate: 2016-12-15 18:49:05 +0100 (jue, 15 dic 2016) $<br>
 * $LastChangedBy: jgonzalez $
 */ 
@SuppressWarnings("deprecation")
public class ActivityQrCodeScanner extends Activity {

	/** When getting a barcode or QR, it will be returned in the intent under this porperty. */
	public static final String INTENT_EXTRA_CODE_READ_RESULT = "code";
	/** When getting a barcode or QR, it will be returned in the intent under this porperty. */
	public static final String INTENT_EXTRA_CODE_READ_TYPE = "code_type";
	
	/** The request code used when calling QR reader. */
	public static final int INTENT_EXTRA_CODE_READER_REQUESTCODE = 6969;

	
	//Loads the ZBAR native Android libraries
	static {
		System.loadLibrary("iconv");
	}

	/**
	 * A safe way to get an instance of the Camera object.
	 */
	public static Camera getCameraInstance() {
		Camera c = null;
		try {
			c = Camera.open();			
		} catch (Exception e) {
		}
		return c;
	}

	// Mimic continuous auto-focusing
	private AutoFocusCallback mAutoFocusCallback = new AutoFocusCallback() {

		@Override
		public void onAutoFocus(boolean success, Camera camera) {
			mAutoFocusHandler.postDelayed(mDoAutoFocus, 1000);
		}
	};

	private Handler mAutoFocusHandler;

	private Camera mCamera;
	private Runnable mDoAutoFocus = new Runnable() {

		@Override
		public void run() {
			if (mPreviewing) {
				mCamera.autoFocus(mAutoFocusCallback);
			}
		}

	};

	private ImageScanner mImageScanner;
	private CameraPreview mPreview;

	private PreviewCallback mPreviewCallback = new PreviewCallback() {

		@Override
		public void onPreviewFrame(byte[] data, Camera camera) {
			int result;
			SymbolSet syms;
			Image barcode;
			Camera.Parameters parameters = camera.getParameters();
			Size size = parameters.getPreviewSize();

			barcode = new Image(size.width, size.height, "Y800");
			barcode.setData(data);
						
			//Configure the barcode types to look for:
			mImageScanner.setConfig(0, Config.ENABLE, 0); //Disable all the Symbols
			mImageScanner.setConfig(Symbol.EAN13, Config.ENABLE, 1); //Only scan for this type
			
			//Slows the frame, but gets more accuracy
			mImageScanner.setConfig(0, Config.X_DENSITY, 1);
			mImageScanner.setConfig(0, Config.Y_DENSITY, 1);			
			
			result = mImageScanner.scanImage(barcode);
			if (result != 0) {
				mPreviewing = false;
				mCamera.setPreviewCallback(null);
				mCamera.stopPreview();

				syms = mImageScanner.getResults();
				if (!syms.isEmpty()) {
					vibrate();
					playSound();
					Symbol scanResult = syms.iterator().next();
					onQrCodeScanned(scanResult.getData(), scanResult.getType());
				}
			}
		}

	};

	private boolean mPreviewing = true;

	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_qr_code_scanner);

		FrameLayout preview;

		mAutoFocusHandler = new Handler();
		mCamera = getCameraInstance();

		mImageScanner = new ImageScanner();
		
		mPreview = new CameraPreview(this, mCamera, mPreviewCallback, mAutoFocusCallback);

		preview = (FrameLayout) findViewById(R.id.cameraPreview);
		preview.addView(mPreview, 0);
	}

	@Override
	public void onPause() {
		super.onPause();
		releaseCamera();
		finish(); //We close the view to release any other resource.
	}

	private void onQrCodeScanned(String code, int codeType) {
		//We return the result of the scan.
		Intent intent = new Intent();
		intent.putExtra(INTENT_EXTRA_CODE_READ_RESULT, code);
		intent.putExtra(INTENT_EXTRA_CODE_READ_TYPE, getCodeTypeName(codeType));
		setResult(Activity.RESULT_OK, intent);
		finish();
	}
	
	/**
	 * Get the code type name of the scanned code.
	 * 
	 * @param codeType
	 * @return
	 */
	private String getCodeTypeName(int codeType) {
		String codeTypeName = null;
		
		if(codeType==Symbol.CODE128){
			codeTypeName = "CODE128";
		}else if(codeType==Symbol.CODE39){
			codeTypeName = "CODE39";
		}else if(codeType==Symbol.CODE93){
			codeTypeName = "CODE93";
		}else if(codeType==Symbol.DATABAR){
			codeTypeName = "DATABAR";
		}else if(codeType==Symbol.DATABAR_EXP){
			codeTypeName = "DATABAR_EXP";
		}else if(codeType==Symbol.EAN13){
			codeTypeName = "EAN13";
		}else if(codeType==Symbol.EAN8){
			codeTypeName = "EAN8";
		}else if(codeType==Symbol.I25){
			codeTypeName = "I25";
		}else if(codeType==Symbol.ISBN10){
			codeTypeName = "ISBN10";
		}else if(codeType==Symbol.ISBN13){
			codeTypeName = "ISBN13";
		}else if(codeType==Symbol.PARTIAL){
			codeTypeName = "PARTIAL";
		}else if(codeType==Symbol.PDF417){
			codeTypeName = "PDF417";
		}else if(codeType==Symbol.QRCODE){
			codeTypeName = "QRCODE";
		}else if(codeType==Symbol.UPCA){
			codeTypeName = "UPCA";
		}else if(codeType==Symbol.UPCE){
			codeTypeName = "UPCE";
		}else if(codeType==Symbol.CODABAR){
			codeTypeName = "CODABAR";
		}else if(codeType==Symbol.NONE){
			codeTypeName = "NONE";
		}
		
		return codeTypeName;
	}

	/** 
	 * Plays a sound when a code is read.
	 */
	private void playSound() {
		ToolBox.media_soundPlay(this, R.raw.beep);		
	}
	
	private void vibrate() {
		ToolBox.device_vibrate(this, 1000);
	}

	private void releaseCamera() {
		if (mCamera != null) {
			mPreviewing = false;
			mCamera.setPreviewCallback(null);
			mCamera.release();
			mCamera = null;
		}
	}	

}