package com.colectivosvip.clubempleadosineco.sqllite;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.location.Location;
import android.text.format.DateFormat;
import android.util.Log;
import es.javocsoft.android.lib.toolbox.ToolBox;
import es.javocsoft.android.lib.toolbox.db.DBHelper;
import es.javocsoft.android.lib.toolbox.db.DBTable;
import com.colectivosvip.clubempleadosineco.Constants;
import com.colectivosvip.clubempleadosineco.sqllite.tables.DBLocation;

/**
 * Class that holds SQLite DB operational methods.
 * 
 * @author JavocSoft, 2015
 * @version 1.0<br>
 * $Rev: 820 $<br>
 * $LastChangedDate: 2015-11-30 16:55:29 +0100 (lun, 30 nov 2015) $<br>
 * $LastChangedBy: admin $
 *
 */
public class DatabaseHelper extends DBHelper<DatabaseSQLite> {

	public static final String LOCATION_TS_STRING_FORMAT = "yyyy-MM-dd HH:MM:ss";
	
	public static final String DB_TABLE_LOCATION = "location";
	public static final String DB_TABLE_LOCATION_ID = "id";
	public static final String DB_TABLE_LOCATION_LAT = "lat";
	public static final String DB_TABLE_LOCATION_LNG = "lng";	
	public static final String DB_TABLE_LOCATION_CITY = "city";
	public static final String DB_TABLE_LOCATION_COUNTRY = "country";
	public static final String DB_TABLE_LOCATION_COUNTRY_CODE = "countryCode";
	public static final String DB_TABLE_LOCATION_POSTAL_CODE = "postalCode";	
	public static final String DB_TABLE_LOCATION_TS = "ts";
	public static final String DB_TABLE_LOCATION_TSSTRING = "tsString";
	public static final String DB_TABLE_LOCATION_NOTREMIND = "notRemind";
	
	
	
	public DatabaseHelper(DatabaseSQLite mDbHelper) {
		super(mDbHelper);	
	}

	
	/**
	 * Inserts a new location in the DB.
	 * 
	 * @param context	Optional. If set the City, Country, Country 
	 * 					code and postal code.
	 * @param location	The location to save.
	 * @param deleteAll	If set to TRUE all content is deleted before insert.
	 * @return	The inserted rowId or DB_ERROR (-1) if there is an error.
	 */
	public synchronized long insertLocation(Context context, Location location, boolean deleteAll) {
		long res = DB_ERROR;
		
		if(location!=null){
			open();	
			
			//We delete previous locations
			if(deleteAll) {
				if(deleteAll(DB_TABLE_LOCATION)<1){
					Log.w(Constants.TAG, "Rows could not be deleted from '" + DB_TABLE_LOCATION + "' table.");
				}
			}
			
			//Insert the new location
			long now = new Date().getTime();
			String tsString = DateFormat.format(LOCATION_TS_STRING_FORMAT, now).toString();
			
			DBLocation loc = new DBLocation(DB_TABLE_LOCATION);
			loc.lat = location.getLatitude();
			loc.lng = location.getLongitude();			
			if(context!=null) {
				loc.city = ToolBox.location_addressInfo(context, ToolBox.LOCATION_INFO_TYPE.CITY, location.getLatitude(), location.getLongitude());
				loc.country = ToolBox.location_addressInfo(context, ToolBox.LOCATION_INFO_TYPE.COUNTRY, location.getLatitude(), location.getLongitude());
				loc.countryCode = ToolBox.location_addressInfo(context, ToolBox.LOCATION_INFO_TYPE.COUNTRY_CODE, location.getLatitude(), location.getLongitude());
				loc.postalCode = ToolBox.location_addressInfo(context, ToolBox.LOCATION_INFO_TYPE.POSTAL_CODE, location.getLatitude(), location.getLongitude());
			}
			loc.ts = now;
			loc.tsString = tsString;
			loc.notRemind = 0; //Default always remind.
			
			res = insert(loc, null);				
			
			close();
		}
		
		return res;
	}
	
	/**
	 * Updates the specified location in the DB.
	 * 
	 * @param context		Optional. If set the City, Country, Country 
	 * 						code and postal code.
	 * @param dbLocation	The location to update.
	 * @param newLocation	The new location info.
	 * @return	TRUE if is done without problems.
	 */
	public synchronized boolean updateLocation(Context context, DBLocation dbLocation, Location newLocation) {
		boolean res = false;
		
		if(dbLocation!=null && newLocation!=null){
			open();	
			
			long now = new Date().getTime();
			String tsString = DateFormat.format(LOCATION_TS_STRING_FORMAT, now).toString();
			dbLocation.ts = now;
			dbLocation.tsString = tsString;
			dbLocation.lat = newLocation.getLatitude();
			dbLocation.lng = newLocation.getLongitude();
			
			ContentValues contentValues = new ContentValues();
			contentValues.put(DB_TABLE_LOCATION_LAT, dbLocation.lat);
			contentValues.put(DB_TABLE_LOCATION_LNG, dbLocation.lng);
			if(context!=null) {
				contentValues.put(DB_TABLE_LOCATION_CITY, ToolBox.location_addressInfo(context, ToolBox.LOCATION_INFO_TYPE.CITY, newLocation.getLatitude(), newLocation.getLongitude()));
				contentValues.put(DB_TABLE_LOCATION_COUNTRY, ToolBox.location_addressInfo(context, ToolBox.LOCATION_INFO_TYPE.COUNTRY, newLocation.getLatitude(), newLocation.getLongitude()));
				contentValues.put(DB_TABLE_LOCATION_COUNTRY_CODE, ToolBox.location_addressInfo(context, ToolBox.LOCATION_INFO_TYPE.COUNTRY_CODE, newLocation.getLatitude(), newLocation.getLongitude()));
				contentValues.put(DB_TABLE_LOCATION_POSTAL_CODE, ToolBox.location_addressInfo(context, ToolBox.LOCATION_INFO_TYPE.POSTAL_CODE, newLocation.getLatitude(), newLocation.getLongitude()));
			}
			contentValues.put(DB_TABLE_LOCATION_TS, dbLocation.ts);
			contentValues.put(DB_TABLE_LOCATION_TSSTRING, dbLocation.tsString);
			contentValues.put(DB_TABLE_LOCATION_NOTREMIND, dbLocation.notRemind);
			
			long resUpdate = updateWhere(DB_TABLE_LOCATION, DB_TABLE_LOCATION_ID + "=?", new String[]{String.valueOf(dbLocation.id)}, contentValues);
			if(resUpdate==1){
				res = true;
			}			
			close();
		}
		
		return res;
	}
	
	/**
	 * Updates the specified location remind option in the DB.
	 * 
	 * @param dbLocation	The location to update.
	 * @param remindOption	The option. Set to 1 to do not remind the location anymore.
	 * @return	TRUE if is done without problems.
	 */
	public synchronized boolean updateLocationRemindOption(DBLocation dbLocation, int remindOption) {
		boolean res = false;
		
		if(dbLocation!=null){
			open();	
			
			ContentValues contentValues = new ContentValues();
			contentValues.put(DB_TABLE_LOCATION_LAT, dbLocation.lat);
			contentValues.put(DB_TABLE_LOCATION_LNG, dbLocation.lng);
			contentValues.put(DB_TABLE_LOCATION_CITY, dbLocation.city);
			contentValues.put(DB_TABLE_LOCATION_COUNTRY, dbLocation.country);
			contentValues.put(DB_TABLE_LOCATION_COUNTRY_CODE, dbLocation.countryCode);
			contentValues.put(DB_TABLE_LOCATION_POSTAL_CODE, dbLocation.postalCode);			
			contentValues.put(DB_TABLE_LOCATION_TS, dbLocation.ts);
			contentValues.put(DB_TABLE_LOCATION_TSSTRING, dbLocation.tsString);
			contentValues.put(DB_TABLE_LOCATION_NOTREMIND, remindOption);
			
			long resUpdate = updateWhere(DB_TABLE_LOCATION, DB_TABLE_LOCATION_ID + "=?", new String[]{String.valueOf(dbLocation.id)}, contentValues);
			if(resUpdate==1){
				res = true;
			}			
			close();
		}
		
		return res;
	}
	
	/**
	 * Gets a location given its Id.
	 *  
	 * @param id
	 * @return
	 */
	public synchronized DBLocation getLocation(long id){
	    DBLocation loc = null;

	    open();	    
	    Cursor c = get(DB_TABLE_LOCATION, DBTable.getColumnsAsArray(DBLocation.class), DB_TABLE_LOCATION_ID + "=" + id, null, null, null, null, null);
	    if(c.moveToFirst()) {
	    	loc = new DBLocation(DB_TABLE_LOCATION);
	    	loc.id = c.getInt(c.getColumnIndex(DB_TABLE_LOCATION_ID));	    	
	    	loc.lat = c.getDouble(c.getColumnIndex(DB_TABLE_LOCATION_LAT));
	    	loc.lng = c.getDouble(c.getColumnIndex(DB_TABLE_LOCATION_LNG));	    	
	    	loc.city = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_CITY));
	    	loc.country = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_COUNTRY));
	    	loc.countryCode = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_COUNTRY_CODE));
	    	loc.postalCode = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_POSTAL_CODE));	    	
	    	loc.ts = c.getLong(c.getColumnIndex(DB_TABLE_LOCATION_TS));
	    	loc.tsString = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_TSSTRING));
	    	loc.notRemind = c.getInt(c.getColumnIndex(DB_TABLE_LOCATION_NOTREMIND));
	    }
	    c.close();	    
	    close();

	    return loc;
	}
		
	/**
	 * Gets the last acquired location.
	 * 
	 * @return
	 */
	public synchronized DBLocation getLastLocation() {
		DBLocation loc = null;
		
		open();
		Cursor c = get(DB_TABLE_LOCATION, null, null, null, null, null, DB_TABLE_LOCATION_TS, QUERY_ORDER.DESC);		
		if(c.moveToFirst()) {
	    	loc = new DBLocation("location");
	    	loc.id = c.getInt(c.getColumnIndex(DB_TABLE_LOCATION_ID));	    	
	    	loc.lat = c.getDouble(c.getColumnIndex(DB_TABLE_LOCATION_LAT));
	    	loc.lng = c.getDouble(c.getColumnIndex(DB_TABLE_LOCATION_LNG));
	    	loc.city = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_CITY));
	    	loc.country = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_COUNTRY));
	    	loc.countryCode = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_COUNTRY_CODE));
	    	loc.postalCode = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_POSTAL_CODE));
	    	loc.ts = c.getLong(c.getColumnIndex(DB_TABLE_LOCATION_TS));
	    	loc.tsString = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_TSSTRING));
	    	loc.notRemind = c.getInt(c.getColumnIndex(DB_TABLE_LOCATION_NOTREMIND));
	    }
	    c.close();
	    close();

	    return loc;
	}	
	
	/**
	 * Returns all registered locations.
	 * 
	 * @return
	 */
	public synchronized List<DBLocation> getLocations() {
		List<DBLocation> res = new ArrayList<DBLocation>();		
		DBLocation loc = null;
		
		open();
		Cursor c = get(DB_TABLE_LOCATION, null, null, null, null, null, DB_TABLE_LOCATION_TS, QUERY_ORDER.DESC);		
		while(c.moveToNext()) {			
			loc = new DBLocation("location");
	    	loc.id = c.getInt(c.getColumnIndex(DB_TABLE_LOCATION_ID));	    	
	    	loc.lat = c.getDouble(c.getColumnIndex(DB_TABLE_LOCATION_LAT));
	    	loc.lng = c.getDouble(c.getColumnIndex(DB_TABLE_LOCATION_LNG));
	    	loc.city = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_CITY));
	    	loc.country = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_COUNTRY));
	    	loc.countryCode = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_COUNTRY_CODE));
	    	loc.postalCode = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_POSTAL_CODE));
	    	loc.ts = c.getLong(c.getColumnIndex(DB_TABLE_LOCATION_TS));
	    	loc.tsString = c.getString(c.getColumnIndex(DB_TABLE_LOCATION_TSSTRING));
	    	loc.notRemind = c.getInt(c.getColumnIndex(DB_TABLE_LOCATION_NOTREMIND));
	    	
	    	res.add(loc);
	    }
	    c.close();
	    close();

	    return res;
	}
	
}
